#ifndef AHLGREN_BOTTOM
#define AHLGREN_BOTTOM

#include "functor.h"
#include "functor_tbl.h"
#include "program.h"
#include "mode.h"
#include "unification.h"

#include <map>
#include <list>
#include <exception>
#include <iterator> // std::next

namespace lp {

	// Exception class for filled bottom clause
	struct bottom_max_length : std::out_of_range {
		bottom_max_length() : std::out_of_range("Bottom Clause Filled") {}
	};

	void reset_mask_bottom();
	void mask_bottom(Functor& bottom, const bitstring& mask);
	void unmask_bottom(Functor& bottom);
	void print_candidate(std::ostream& os, const Functor& c);

	bool is_subset(const bitstring& p, const bitstring& q);

	/* Keep track of:
	- Mapping from terms to variables (termindex)
	- Instantiated Terms (instantiated)
	for each instantiated term
	for each mode
	1. Instantiate all possible +types
	2. Query to get all -types
	3. Add if new
	*/

	class Termindex {
	public:
		//typedef Functor::var_index_type index_type;
		// Map terms to variables
		typedef list<id_type> var_range;
		typedef map<Functor,var_range> mapping_type;
		// Store all instantiated terms (they have been outputted by some literal)
		typedef set<Functor> inst_type; // instantiated terms
		typedef set<id_type> inst_var_type; // instantiated variables
		// Iterators
		typedef mapping_type::const_iterator map_iterator;
		typedef inst_type::const_iterator inst_iterator;
		typedef Mode::const_iterator pm_iter;
		typedef var_range::const_iterator r_iterator; // range iterator

		// Construct empty term index
		Termindex() { }

		// Clear termindex
		void clear();
		// Index a term or get index of revisit (make a unique variable name)
		id_type index(const Functor& f);
		id_type multi_index(const Functor& f); // variable splitting version
		// Register term/variable instance (output)
		void instance(const Functor& trm, id_type var);
		// Instantiate all types mt
		void instantiate(const Functor& trm, const Functor& lifted, const Mode& m, const Mode::marker mt);
		void instantiate_inputs(const Functor& trm, const Functor& lifted, const Mode& m);
		void instantiate_outputs(const Functor& trm, const Functor& lifted, const Mode& m);
		// Check if term is instantiated
		bool is_instantiated(const Functor& trm) const;
		// Check if variable is instantiated
		bool is_instantiated(id_type var) const;

		// Update termindex for a new generation
		void update_iteration()
		{
			std::move(next_instantiated.begin(),next_instantiated.end(),inserter(instantiated,instantiated.begin()));
			next_instantiated.clear();
			std::move(next_var_instantiated.begin(),next_var_instantiated.end(),
				inserter(var_instantiated,var_instantiated.begin()));
			next_var_instantiated.clear();
		}

		// Get termindex range, assumes term f has been indexed
		pair<r_iterator,r_iterator> range(const Functor& f) const
		{
			auto at = mapping.find(f);
			assert( at != mapping.end() );
			return make_pair(at->second.begin(),at->second.end());
		}

		// Iterators (const versions only)
		inst_iterator inst_begin() const { return instantiated.begin(); }
		inst_iterator inst_end() const { return instantiated.end(); }

		void print(ostream& os) const;
	protected:
		mapping_type mapping;
		inst_type instantiated; // instantiated terms
		inst_var_type var_instantiated; // variables that are instantiated
		// Instantiated but will not be used until next iteration:
		inst_type next_instantiated;
		inst_var_type next_var_instantiated;
		// vector<const Functor*> last_instantiated; // terms instantiated by current iteration
	private:
		Termindex(const Termindex&);
		Termindex& operator=(const Termindex&);
	};


	//======================== Inline Definitions ========================//

	inline ostream& operator<<(ostream& os, const Termindex& t) {
		t.print(os);
		return os;
	}

	inline void Termindex::clear() 
	{
		mapping.clear();
		instantiated.clear();
		var_instantiated.clear();
		next_instantiated.clear();
		next_var_instantiated.clear();
	}

	inline void Termindex::instantiate(const Functor& trm, const Functor& lifted, const Mode& mode, Mode::marker mtype)
	{
		for (auto m = mode.begin(); m != mode.end(); ++m) {
			if (m->iotype == mtype) {
				this->instance(*trm.get(m->pos),lifted.get(m->pos)->id());
			}
		}
	}

	inline void Termindex::instantiate_inputs(const Functor& trm, const Functor& lifted, const Mode& mode)
	{
		return this->instantiate(trm,lifted,mode,Mode::input);
	}

	inline void Termindex::instantiate_outputs(const Functor& trm, const Functor& lifted, const Mode& mode)
	{
		return this->instantiate(trm,lifted,mode,Mode::output);
	}

	inline bool Termindex::is_instantiated(const Functor& trm) const
	{
		return instantiated.find(trm) != instantiated.end();
	}

	inline bool Termindex::is_instantiated(id_type var) const
	{
		return var_instantiated.find(var) != var_instantiated.end(); 
	}

	inline id_type Termindex::index(const Functor& f) 
	{
		auto at = mapping.find(f);
		if (at == mapping.end()) {
			// create singleton value
			const id_type vn = Functor::unique_index();
			DEBUG_TRACE( cerr << "termindex: indexing " << f << " -> " << Functor::get_data(vn) << "\n" );
			at = mapping.insert(mapping_type::value_type(f,var_range(1,vn))).first;
		}
		assert( at->second.size() == 1 );
		return at->second.front();
	}

	inline id_type Termindex::multi_index(const Functor& f)
	{
		const id_type vn = Functor::unique_index();
		auto at = mapping.find(f);
		if (at == mapping.end()) {
			// create singleton value
			DEBUG_TRACE( cerr << "termindex: multi_indexing new " << f << " -> " << Functor::get_data(vn) << "\n" );
			at = mapping.insert(mapping_type::value_type(f,var_range(1,vn))).first;
		} else {
			// Insert new value if not already present
			DEBUG_TRACE( cerr << "termindex: multi_indexing old " << f << " -> " << Functor::get_data(vn) << "\n" );
			at->second.push_back(vn);
		}
		return vn;
	}

	inline void Termindex::instance(const Functor& trm, id_type var)
	{
		assert( trm.is_ground() );
		next_instantiated.insert(trm);
		next_var_instantiated.insert(var);
	}

	// Lift ground truth using termindex (no variable splitting)
	inline void lift(
		Functor& f,
		Mode::const_iterator pm_beg,
		Mode::const_iterator pm_end,
		Termindex& termindex)
	{
		// Grab all placemarker positions and lift them to termindex variable
		for_each(pm_beg, pm_end, [&](const Mode::pmarker & p) {
			if (p.iotype == Mode::input || p.iotype == Mode::output) {
				Functor* term = f.get(p.pos);
				const id_type var = termindex.index(*term);
				// Lift it
				DEBUG_TRACE(cerr << "lifting " << *term << " -> ");
				*term = Functor(var);
				DEBUG_TRACE(cerr << *term << "\n");
			}
		});
	}

	inline void variable_split_head(
		Functor& f,
		Mode::const_iterator pm_beg,
		const Mode& mode,
		Termindex& termindex,
		list<Functor>& botv,
		vector<Mode>& modes,
		set<const Functor*,functor_ptr_less>& botv_memo) 
	{
		assert( botv.empty() );
		// Grab all placemarker positions and lift them to termindex variable
		for_each(pm_beg, mode.end(), [&](const Mode::pmarker & p) {
			if (p.iotype == Mode::input || p.iotype == Mode::output) {
				// Split variable
				Functor* term = f.get(p.pos);
				const id_type var = termindex.multi_index(*term);
				// Add equalities
				auto r = termindex.range(*term); // is non-empty since we just called multi_index
				for ( ; r.first != r.second; ++r.first) {
					if (*r.first == var) continue; // skip same
					botv.push_back(Functor(equal2_id, new Functor(var), new Functor(*r.first)));
				}
				// Register instance
				if (p.iotype == Mode::input) {
					termindex.instance(*term,var); // term is now variable
				}
				// Lift term to variable
				*term = Functor(var);
			}
		});
		botv.push_front(f);
		botv_memo.insert(&botv.front());
		modes.push_back(mode);
	}

	// Lift ground truth using termindex (variable splitting)
	/* When lifting body literal two things can happen for each +/- type:
	1) Output variable is lifted. This simply means splitting the variable and adding equalities.
	2) Input  variable is lifted. This case is more complicated, as the input variable may
	catch output from a number of different previous outputs. For each output, one
	instance of the literal needs to be created.					
	*/
	inline void variable_split(
		Functor& build,
		Mode::const_iterator pm_curr,
		const Mode& mode,
		Termindex& termindex,
		list<Functor>& botv,
		vector<Mode>& modes,
		set<const Functor*,functor_ptr_less>& botv_memo,
		const parameters& params)
	{
		DEBUG_TRACE( cerr << "variable_split(), current build: " << build << "\n");
		if (pm_curr == mode.end()) {
			// Finished
			DEBUG_TRACE( cerr << "variable_split() finished, adding bottom literal: " << build << "\n");
			modes.push_back(mode);
			botv.push_back(build);
			botv_memo.insert(&botv.back());
			// Check bsize. Note: botv.size() also contains head, so botv.size-1 is no of body literals
			if (int(botv.size())-1 >= params.force_int(parameters::bsize)) throw bottom_max_length();
			return;
		} else if (pm_curr->iotype == Mode::output) {
			// Split variable
			Functor* term = build.get(pm_curr->pos);
			const id_type var = termindex.multi_index(*term);
			DEBUG_TRACE(cerr << "output type, term: " << *term << " -> " << Functor::get_data(var) << "\n");
			// Add equalities
			auto r = termindex.range(*term); // is non-empty since we just called multi_index
			for ( ; r.first != r.second; ++r.first) {
				if (*r.first == var) continue; // skip same
				auto after_head = (botv.empty() ? botv.begin() : ++botv.begin());
				//cerr << "splitvars: inserting " << Functor(equal2_id, new Functor(var), new Functor(*r.first)) << "\n";
				botv.insert(after_head,Functor(equal2_id, new Functor(var), new Functor(*r.first)));
				// TODO: if bottom_max_length is reached, many added equalities should be removed again
				// TODO: this can be precalculated before entering the loop
				// Check bsize. Note: botv.size() also contains head, so botv.size-1 is no of body literals
				//if (int(botv.size())-1 >= params.bsize) throw bottom_max_length();
			}
			// Register instance
			termindex.instance(*term,var);
			// Lift term to variable
			*term = Functor(var);
			// continue
			return variable_split(build,++pm_curr,mode,termindex,botv,modes,botv_memo,params);
		} else if (pm_curr->iotype == Mode::input) {
			// Substitute term with all possible input variables
			DEBUG_TRACE(cerr << "input type, term: " << *build.get(pm_curr->pos) << "\n");
			auto p = termindex.range(*build.get(pm_curr->pos));
			assert( p.first != p.second );
			for ( ; p.first != p.second; ++p.first) {
				// Check that variable is instantiated
				if (termindex.is_instantiated(*p.first)) {
					Functor sub_build = build;
					// Lift
					*sub_build.get(pm_curr->pos) = *p.first;
					// Continue building sub_build
					variable_split(sub_build,std::next(pm_curr),mode,termindex,botv,modes,botv_memo,params);
				}
			}
			return;
		} else {
			// Ground type, don't change anything
			DEBUG_TRACE(cerr << "ground type: " << *build.get(pm_curr->pos) << "\n");
			return variable_split(build,++pm_curr,mode,termindex,botv,modes,botv_memo,params);
		}
	}

	void add_varsplit_equalities(
		const Functor& instance, 
		const Functor& linstance, 
		const Mode& mode,
		const Termindex& termindex,
		list<Functor>& botv,
		bool is_head = false);


}

#endif

